#!/bin/bash
#
# TheiaCast Client Installation Script
# Supports: Ubuntu Desktop/Server, Raspberry Pi, with or without displays
#

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Configuration
INSTALL_DIR="/opt/theiacast-client"
SERVICE_NAME="theiacast-client"
SERVICE_FILE="/etc/systemd/system/${SERVICE_NAME}.service"

echo -e "${GREEN}========================================${NC}"
echo -e "${GREEN}  TheiaCast Client Installer${NC}"
echo -e "${GREEN}========================================${NC}"
echo ""

# Check if running as root
if [ "$EUID" -ne 0 ]; then
  echo -e "${RED}Error: This script must be run as root${NC}"
  echo "Please run: sudo bash $0"
  exit 1
fi

# Detect actual user (not root)
ACTUAL_USER=${SUDO_USER:-$(who am i | awk '{print $1}')}
ACTUAL_USER=${ACTUAL_USER:-$(logname 2>/dev/null)}
USER_HOME=$(eval echo ~${ACTUAL_USER})

if [ -z "$ACTUAL_USER" ] || [ "$ACTUAL_USER" = "root" ]; then
  echo -e "${RED}Error: Cannot detect non-root user${NC}"
  echo "Please run as: sudo bash $0 (not as root user)"
  exit 1
fi

echo -e "${BLUE}Installing for user:${NC} $ACTUAL_USER"
echo -e "${BLUE}User home:${NC} $USER_HOME"
echo ""

# Detect OS version
if [ -f /etc/os-release ]; then
    . /etc/os-release
    OS_VERSION=$VERSION_ID
    OS_NAME=$NAME
    echo -e "${BLUE}Detected OS:${NC} $NAME $VERSION"
else
    OS_VERSION="unknown"
    OS_NAME="unknown"
    echo -e "${YELLOW}⚠️  Could not detect OS version${NC}"
fi

# Detect system architecture
ARCH=$(uname -m)
echo -e "${BLUE}Architecture:${NC} ${ARCH}"

# Detect if Raspberry Pi
IS_RASPBERRY_PI=false
if [[ "$OS_NAME" == *"Raspbian"* ]] || [[ "$OS_NAME" == *"Raspberry"* ]] || [ -f /proc/device-tree/model ]; then
    if [ -f /proc/device-tree/model ]; then
        PI_MODEL=$(cat /proc/device-tree/model 2>/dev/null)
        if [[ "$PI_MODEL" == *"Raspberry Pi"* ]]; then
            IS_RASPBERRY_PI=true
            echo -e "${BLUE}Device:${NC} Raspberry Pi detected ($PI_MODEL)"
        fi
    fi
fi
echo ""

# ============================================
# STEP 1: Install Node.js
# ============================================
echo -e "${BLUE}[1/6] Checking Node.js...${NC}"
if ! command -v node &> /dev/null; then
  echo "Installing Node.js 20.x..."
  curl -fsSL https://deb.nodesource.com/setup_20.x | bash -
  apt-get install -y nodejs
fi

NODE_VERSION=$(node -v)
echo -e "${GREEN}✓${NC} Node.js ${NODE_VERSION} installed"
echo ""

# ============================================
# STEP 2: Detect Display Hardware
# ============================================
echo -e "${BLUE}[2/6] Detecting display hardware...${NC}"
HAS_DISPLAY=false
USE_XVFB=false
SKIP_X_SETUP=false

if [ "$IS_RASPBERRY_PI" = true ]; then
    # Raspberry Pi with Desktop OS - assume GUI environment already configured
    SKIP_X_SETUP=true
    echo -e "${GREEN}✓${NC} Raspberry Pi detected - using existing desktop environment"
elif lspci 2>/dev/null | grep -iq "vga\|3d\|display"; then
    HAS_DISPLAY=true
    GPU_INFO=$(lspci | grep -i "vga\|3d\|display" | head -1)
    echo -e "${GREEN}✓${NC} Physical display detected: $GPU_INFO"
else
    USE_XVFB=true
    echo -e "${YELLOW}ℹ${NC}  No physical display detected - will use virtual display (Xvfb)"
fi
echo ""

# ============================================
# STEP 3: Install Display Dependencies
# ============================================
echo -e "${BLUE}[3/6] Installing display dependencies...${NC}"

# Update package list
apt-get update -qq

if [ "$SKIP_X_SETUP" = true ]; then
    echo "Skipping X server setup (Raspberry Pi with existing desktop environment)"
    echo -e "${GREEN}✓${NC} Using existing desktop environment"

elif [ "$USE_XVFB" = true ]; then
    echo "Installing Xvfb (virtual display) for headless mode..."
    apt-get install -y xvfb

    # Install minimal Chrome dependencies
    if [[ "$OS_VERSION" == "24.04" ]] || [[ "$OS_VERSION" > "24.04" ]]; then
        CHROME_DEPS=(
            ca-certificates fonts-liberation libasound2t64 libatk-bridge2.0-0t64
            libatk1.0-0t64 libc6 libcairo2 libcups2t64 libdbus-1-3 libexpat1
            libfontconfig1 libgbm1 libgcc-s1 libglib2.0-0t64 libgtk-3-0t64
            libnspr4 libnss3 libpango-1.0-0 libpangocairo-1.0-0 libstdc++6
            libx11-6 libx11-xcb1 libxcb1 libxcomposite1 libxcursor1
            libxdamage1 libxext6 libxfixes3 libxi6 libxrandr2 libxrender1
            libxss1 libxtst6 lsb-release wget xdg-utils
        )
    else
        CHROME_DEPS=(
            ca-certificates fonts-liberation libasound2 libatk-bridge2.0-0
            libatk1.0-0 libc6 libcairo2 libcups2 libdbus-1-3 libexpat1
            libfontconfig1 libgbm1 libgcc-s1 libglib2.0-0 libgtk-3-0
            libnspr4 libnss3 libpango-1.0-0 libpangocairo-1.0-0 libstdc++6
            libx11-6 libx11-xcb1 libxcb1 libxcomposite1 libxcursor1
            libxdamage1 libxext6 libxfixes3 libxi6 libxrandr2 libxrender1
            libxss1 libxtst6 lsb-release wget xdg-utils
        )
    fi
    apt-get install -y "${CHROME_DEPS[@]}"
    echo -e "${GREEN}✓${NC} Xvfb and dependencies installed"

elif [ "$HAS_DISPLAY" = true ]; then
    echo "Installing X server and dependencies..."

    # Determine package names based on Ubuntu version (24.04+ uses t64 variants)
    if [[ "$OS_VERSION" == "24.04" ]] || [[ "$OS_VERSION" > "24.04" ]]; then
        echo "Using time64 packages for Ubuntu 24.04+"
        CHROME_DEPS=(
            ca-certificates fonts-liberation libasound2t64 libatk-bridge2.0-0t64
            libatk1.0-0t64 libc6 libcairo2 libcups2t64 libdbus-1-3 libexpat1
            libfontconfig1 libgbm1 libgcc-s1 libglib2.0-0t64 libgtk-3-0t64
            libnspr4 libnss3 libpango-1.0-0 libpangocairo-1.0-0 libstdc++6
            libx11-6 libx11-xcb1 libxcb1 libxcomposite1 libxcursor1
            libxdamage1 libxext6 libxfixes3 libxi6 libxrandr2 libxrender1
            libxss1 libxtst6 lsb-release wget xdg-utils
        )
    else
        CHROME_DEPS=(
            ca-certificates fonts-liberation libasound2 libatk-bridge2.0-0
            libatk1.0-0 libc6 libcairo2 libcups2 libdbus-1-3 libexpat1
            libfontconfig1 libgbm1 libgcc-s1 libglib2.0-0 libgtk-3-0
            libnspr4 libnss3 libpango-1.0-0 libpangocairo-1.0-0 libstdc++6
            libx11-6 libx11-xcb1 libxcb1 libxcomposite1 libxcursor1
            libxdamage1 libxext6 libxfixes3 libxi6 libxrandr2 libxrender1
            libxss1 libxtst6 lsb-release wget xdg-utils
        )
    fi

    apt-get install -y "${CHROME_DEPS[@]}"

    # Install X server and window manager
    apt-get install -y xserver-xorg x11-xserver-utils xinit openbox xdotool unclutter

    # Detect and install GPU drivers
    if echo "$GPU_INFO" | grep -iq "intel"; then
        echo "Installing Intel graphics drivers..."
        apt-get install -y xserver-xorg-video-intel mesa-utils libgl1-mesa-dri va-driver-all

        # Configure Intel GPU
        mkdir -p /etc/X11/xorg.conf.d
        cat > /etc/X11/xorg.conf.d/20-intel.conf << 'EOF'
Section "Device"
    Identifier "Intel Graphics"
    Driver "modesetting"
    Option "AccelMethod" "glamor"
    Option "DRI" "3"
EndSection
EOF
        modprobe i915 || true

    elif echo "$GPU_INFO" | grep -iq "nvidia"; then
        echo "Installing NVIDIA graphics drivers..."
        apt-get install -y nvidia-driver-535 || apt-get install -y nvidia-driver || true

    elif echo "$GPU_INFO" | grep -iq "amd\|ati\|radeon"; then
        echo "Installing AMD graphics drivers..."
        apt-get install -y xserver-xorg-video-amdgpu mesa-utils libgl1-mesa-dri

    elif echo "$GPU_INFO" | grep -iq "broadcom\|videocore"; then
        echo "Raspberry Pi GPU detected - using default drivers"
        # Raspberry Pi uses VideoCore GPU with built-in drivers

    else
        echo "Installing generic graphics drivers..."
        apt-get install -y mesa-utils libgl1-mesa-dri
    fi

    # Add user to video/input/render groups
    usermod -a -G video $ACTUAL_USER 2>/dev/null || true
    usermod -a -G input $ACTUAL_USER 2>/dev/null || true
    usermod -a -G render $ACTUAL_USER 2>/dev/null || true

    echo -e "${GREEN}✓${NC} X server and GPU drivers installed"

else
    echo "Installing Xvfb (virtual display) for headless mode..."
    apt-get install -y xvfb
    echo -e "${GREEN}✓${NC} Xvfb installed"
fi
echo ""

# ============================================
# STEP 4: Install Chromium
# ============================================
echo -e "${BLUE}[4/6] Installing Chromium browser...${NC}"

CHROMIUM_EXECUTABLE_PATH=""
if [[ "$ARCH" == "aarch64" || "$ARCH" == "arm64" || "$ARCH" == "armv7l" ]]; then
    # ARM system (Raspberry Pi) - use system chromium
    echo "ARM system - installing system Chromium..."

    if apt-cache show chromium &> /dev/null; then
        apt-get install -y chromium
    elif apt-cache show chromium-browser &> /dev/null; then
        apt-get install -y chromium-browser
    else
        echo -e "${YELLOW}⚠${NC}  Could not find chromium in apt repositories"
    fi

    # Find chromium executable
    if command -v chromium &> /dev/null; then
        CHROMIUM_EXECUTABLE_PATH=$(which chromium)
    elif command -v chromium-browser &> /dev/null; then
        CHROMIUM_EXECUTABLE_PATH=$(which chromium-browser)
    fi

    if [ -n "$CHROMIUM_EXECUTABLE_PATH" ]; then
        echo -e "${GREEN}✓${NC} System Chromium: ${CHROMIUM_EXECUTABLE_PATH}"
    else
        echo -e "${YELLOW}⚠${NC}  Chromium not found - Puppeteer will download it"
    fi
else
    # x64 system - install system Chromium
    echo "x64 system - installing system Chromium..."

    if ! command -v chromium &> /dev/null && ! command -v chromium-browser &> /dev/null; then
        apt-get install -y chromium-browser || apt-get install -y chromium || {
            echo -e "${YELLOW}⚠${NC}  Could not install Chromium - Puppeteer will try to download Chrome on first run"
        }
    fi

    # Find chromium executable
    if command -v chromium &> /dev/null; then
        CHROMIUM_EXECUTABLE_PATH=$(which chromium)
        echo -e "${GREEN}✓${NC} System Chromium: ${CHROMIUM_EXECUTABLE_PATH}"
    elif command -v chromium-browser &> /dev/null; then
        CHROMIUM_EXECUTABLE_PATH=$(which chromium-browser)
        echo -e "${GREEN}✓${NC} System Chromium: ${CHROMIUM_EXECUTABLE_PATH}"
    else
        echo -e "${YELLOW}⚠${NC}  Chromium not installed - Puppeteer will download Chrome on first run"
    fi
fi
echo ""

# ============================================
# STEP 5: Install TheiaCast Client
# ============================================
echo -e "${BLUE}[5/6] Installing TheiaCast client...${NC}"

# Stop existing service if running
if systemctl is-active --quiet ${SERVICE_NAME}; then
    echo "Stopping existing service..."
    systemctl stop ${SERVICE_NAME}
fi

# Create installation directory
mkdir -p ${INSTALL_DIR}

# Copy files from tarball
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PACKAGE_DIR="$(dirname "$SCRIPT_DIR")"

echo "Copying files from package..."

# Copy dist files
if [ -d "${PACKAGE_DIR}/dist" ]; then
    cp -r ${PACKAGE_DIR}/dist/* ${INSTALL_DIR}/
else
    echo -e "${RED}Error: dist directory not found${NC}"
    exit 1
fi

# Copy package.json
if [ -f "${PACKAGE_DIR}/package.json" ]; then
    cp ${PACKAGE_DIR}/package.json ${INSTALL_DIR}/
fi

# Copy pre-bundled node_modules if available
HAS_PREBUNDLED_DEPS=false
if [ -d "${PACKAGE_DIR}/node_modules" ]; then
    echo "Copying pre-bundled dependencies..."
    cp -r ${PACKAGE_DIR}/node_modules ${INSTALL_DIR}/
    HAS_PREBUNDLED_DEPS=true
fi

# Install dependencies if not pre-bundled
if [ "$HAS_PREBUNDLED_DEPS" = true ]; then
    echo -e "${GREEN}✓${NC} Using pre-bundled dependencies"
elif [ -f "${INSTALL_DIR}/package.json" ]; then
    echo "Installing dependencies from npm..."
    cd ${INSTALL_DIR}
    if [[ "$ARCH" == "aarch64" || "$ARCH" == "arm64" || "$ARCH" == "armv7l" ]]; then
        PUPPETEER_SKIP_CHROMIUM_DOWNLOAD=true npm install --production --no-optional
    else
        npm install --production --no-optional
    fi
fi

# Load existing configuration if available
EXISTING_SERVER_URL=""
EXISTING_DEVICE_ID=""
EXISTING_DEVICE_TOKEN=""

if [ -f "${INSTALL_DIR}/.env" ]; then
    echo ""
    echo -e "${YELLOW}Found existing configuration${NC}"
    EXISTING_SERVER_URL=$(grep "^SERVER_URL=" "${INSTALL_DIR}/.env" 2>/dev/null | cut -d'=' -f2-)
    EXISTING_DEVICE_ID=$(grep "^DEVICE_ID=" "${INSTALL_DIR}/.env" 2>/dev/null | cut -d'=' -f2-)
    EXISTING_DEVICE_TOKEN=$(grep "^DEVICE_TOKEN=" "${INSTALL_DIR}/.env" 2>/dev/null | cut -d'=' -f2-)
fi

# Prompt for configuration
echo ""
echo -e "${BLUE}Configuration:${NC}"
echo -e "${YELLOW}(Press Enter to keep existing values)${NC}"

if [ -n "$EXISTING_SERVER_URL" ]; then
    read -p "Server URL [$EXISTING_SERVER_URL]: " SERVER_URL < /dev/tty
    SERVER_URL=${SERVER_URL:-$EXISTING_SERVER_URL}
else
    read -p "Server URL (e.g., http://192.168.1.100:5001): " SERVER_URL < /dev/tty
fi

if [ -n "$EXISTING_DEVICE_ID" ]; then
    read -p "Device ID [$EXISTING_DEVICE_ID]: " DEVICE_ID < /dev/tty
    DEVICE_ID=${DEVICE_ID:-$EXISTING_DEVICE_ID}
else
    DEFAULT_DEVICE_ID=$(hostname)
    read -p "Device ID [$DEFAULT_DEVICE_ID]: " DEVICE_ID < /dev/tty
    DEVICE_ID=${DEVICE_ID:-$DEFAULT_DEVICE_ID}
fi

if [ -n "$EXISTING_DEVICE_TOKEN" ]; then
    read -p "Device Token [$EXISTING_DEVICE_TOKEN]: " DEVICE_TOKEN < /dev/tty
    DEVICE_TOKEN=${DEVICE_TOKEN:-$EXISTING_DEVICE_TOKEN}
else
    read -p "Device Token: " DEVICE_TOKEN < /dev/tty
fi

# Create .env file
cat > ${INSTALL_DIR}/.env << EOF
SERVER_URL=${SERVER_URL}
DEVICE_ID=${DEVICE_ID}
DEVICE_TOKEN=${DEVICE_TOKEN}
LOG_LEVEL=info
SCREENSHOT_INTERVAL=300000
HEALTH_REPORT_INTERVAL=60000
DISPLAY_WIDTH=1920
DISPLAY_HEIGHT=1080
HEADLESS=false
KIOSK_MODE=true
PUPPETEER_EXECUTABLE_PATH=${CHROMIUM_EXECUTABLE_PATH}
EOF

if [ -z "$CHROMIUM_EXECUTABLE_PATH" ]; then
    echo "PUPPETEER_SKIP_CHROMIUM_DOWNLOAD=false" >> ${INSTALL_DIR}/.env
else
    echo "PUPPETEER_SKIP_CHROMIUM_DOWNLOAD=true" >> ${INSTALL_DIR}/.env
fi

echo -e "${GREEN}✓${NC} Configuration saved"

# Set ownership
chown -R ${ACTUAL_USER}:${ACTUAL_USER} ${INSTALL_DIR}
chmod +x ${INSTALL_DIR}/index.js 2>/dev/null || true

echo -e "${GREEN}✓${NC} Client files installed"
echo ""

# ============================================
# STEP 6: Create Systemd Service
# ============================================
echo -e "${BLUE}[6/6] Creating systemd service...${NC}"

if [ "$SKIP_X_SETUP" = true ]; then
    echo "Configuring for Raspberry Pi (using existing desktop environment)..."

    # Simple systemd service - just run the Node app
    # No dependency on graphical.target to avoid ordering cycles
    # Desktop environment is already running, we just need network
    cat > ${SERVICE_FILE} << EOF
[Unit]
Description=TheiaCast Kiosk Client
After=network-online.target
Wants=network-online.target

[Service]
Type=simple
User=${ACTUAL_USER}
Group=${ACTUAL_USER}
WorkingDirectory=${INSTALL_DIR}
Environment="NODE_ENV=production"
Environment="DISPLAY=:0"
Environment="HOME=${USER_HOME}"
Environment="XAUTHORITY=${USER_HOME}/.Xauthority"
EnvironmentFile=${INSTALL_DIR}/.env
ExecStart=/usr/bin/node ${INSTALL_DIR}/index.js
Restart=always
RestartSec=10
KillMode=control-group
KillSignal=SIGTERM
TimeoutStopSec=10

[Install]
WantedBy=multi-user.target
EOF

    echo -e "${GREEN}✓${NC} Configured for Raspberry Pi desktop"

elif [ "$HAS_DISPLAY" = true ]; then
    echo "Configuring for physical display with X server..."

    # Create openbox autostart
    mkdir -p ${USER_HOME}/.config/openbox
    cat > ${USER_HOME}/.config/openbox/autostart << EOF
# Hide cursor after 1 second
unclutter -idle 1 &

# Disable screen blanking
xset s off
xset -dpms
xset s noblank

# Wait for X server
sleep 2

# Start TheiaCast client
cd ${INSTALL_DIR}
node ${INSTALL_DIR}/index.js >> /tmp/theiacast-client.log 2>&1 &
EOF

    # Create .xinitrc
    cat > ${USER_HOME}/.xinitrc << 'EOF'
#!/bin/bash
exec openbox-session
EOF
    chmod +x ${USER_HOME}/.xinitrc
    chown ${ACTUAL_USER}:${ACTUAL_USER} ${USER_HOME}/.xinitrc
    chown -R ${ACTUAL_USER}:${ACTUAL_USER} ${USER_HOME}/.config/openbox

    # Create systemd service with X server
    cat > ${SERVICE_FILE} << EOF
[Unit]
Description=TheiaCast Kiosk Client with X Server
After=network.target systemd-user-sessions.service

[Service]
Type=simple
User=${ACTUAL_USER}
WorkingDirectory=/home/${ACTUAL_USER}
ExecStart=/usr/bin/startx
Restart=always
RestartSec=10
TTYPath=/dev/tty7
StandardInput=tty
StandardOutput=journal
StandardError=journal

[Install]
WantedBy=multi-user.target
EOF

    echo -e "${GREEN}✓${NC} Configured for physical display"

else
    echo "Configuring for headless mode with Xvfb..."

    # Create Xvfb wrapper script
    cat > ${INSTALL_DIR}/start-xvfb.sh << 'XVFB_EOF'
#!/bin/bash
# Start Xvfb in background if not already running
if ! pgrep -x "Xvfb" > /dev/null; then
    /usr/bin/Xvfb :99 -screen 0 1920x1080x24 -nolisten tcp &
    XVFB_PID=$!
    echo $XVFB_PID > /tmp/xvfb.pid
    sleep 2
fi
XVFB_EOF
    chmod +x ${INSTALL_DIR}/start-xvfb.sh

    # Create systemd service with Xvfb
    cat > ${SERVICE_FILE} << EOF
[Unit]
Description=TheiaCast Kiosk Client (Headless)
After=network.target

[Service]
Type=simple
User=${ACTUAL_USER}
Group=${ACTUAL_USER}
WorkingDirectory=${INSTALL_DIR}
Environment="NODE_ENV=production"
Environment="DISPLAY=:99"
Environment="HOME=${USER_HOME}"
EnvironmentFile=${INSTALL_DIR}/.env
ExecStartPre=/bin/bash ${INSTALL_DIR}/start-xvfb.sh
ExecStart=/usr/bin/node ${INSTALL_DIR}/index.js
ExecStopPost=/bin/sh -c 'if [ -f /tmp/xvfb.pid ]; then kill \$(cat /tmp/xvfb.pid) 2>/dev/null || true; rm -f /tmp/xvfb.pid; fi'
Restart=always
RestartSec=10
KillMode=control-group
KillSignal=SIGTERM
TimeoutStopSec=10

[Install]
WantedBy=multi-user.target
EOF

    echo -e "${GREEN}✓${NC} Configured for headless mode"
fi

# Reload systemd
systemctl daemon-reload

# Enable service
systemctl enable ${SERVICE_NAME}

# Start service
echo "Starting service..."
systemctl start ${SERVICE_NAME}

# Wait and check status
sleep 3

echo ""
if systemctl is-active --quiet ${SERVICE_NAME}; then
    echo -e "${GREEN}========================================${NC}"
    echo -e "${GREEN}  Installation Complete! ✓${NC}"
    echo -e "${GREEN}========================================${NC}"
    echo ""
    echo -e "${BLUE}Service Status:${NC}"
    systemctl status ${SERVICE_NAME} --no-pager -l | head -15
    echo ""
    echo -e "${BLUE}Useful Commands:${NC}"
    echo "  View logs:      sudo journalctl -u ${SERVICE_NAME} -f"
    echo "  Restart:        sudo systemctl restart ${SERVICE_NAME}"
    echo "  Stop:           sudo systemctl stop ${SERVICE_NAME}"
    echo "  Status:         sudo systemctl status ${SERVICE_NAME}"
    echo "  Edit config:    sudo nano ${INSTALL_DIR}/.env"
    echo ""
    if [ "$SKIP_X_SETUP" = true ]; then
        echo -e "${BLUE}Display Mode:${NC} Raspberry Pi Desktop"
        echo "  The browser should now launch in kiosk mode on your display."
    elif [ "$HAS_DISPLAY" = true ]; then
        echo -e "${BLUE}Display Mode:${NC} Physical display (X server on tty7)"
        echo "  The display should now show the browser in kiosk mode."
    else
        echo -e "${BLUE}Display Mode:${NC} Headless (Xvfb virtual display)"
    fi
    echo ""
else
    echo -e "${RED}========================================${NC}"
    echo -e "${RED}  Service Failed to Start!${NC}"
    echo -e "${RED}========================================${NC}"
    echo ""
    echo "Check logs with:"
    echo "  sudo journalctl -u ${SERVICE_NAME} -n 50"
    echo ""
    exit 1
fi
