#!/bin/bash

# Fix systemd ordering cycle on existing TheiaCast installations
# This script updates the service file to remove graphical.target dependency

set -e

RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

SERVICE_FILE="/etc/systemd/system/theiacast-client.service"

echo -e "${BLUE}TheiaCast - Fix Systemd Ordering Cycle${NC}"
echo ""

# Check if service file exists
if [ ! -f "$SERVICE_FILE" ]; then
    echo -e "${RED}✗${NC} Service file not found: $SERVICE_FILE"
    echo "   TheiaCast may not be installed, or the service file is in a different location."
    exit 1
fi

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    echo -e "${RED}✗${NC} This script must be run as root (use sudo)"
    exit 1
fi

echo -e "${YELLOW}Current service file:${NC}"
cat "$SERVICE_FILE"
echo ""

# Check if service has graphical.target dependency
if grep -q "graphical.target" "$SERVICE_FILE"; then
    echo -e "${YELLOW}⚠${NC}  Found graphical.target dependency - this causes ordering cycles"
else
    echo -e "${GREEN}✓${NC} No graphical.target dependency found"
    echo "   Service file appears to be correct already."
    exit 0
fi

# Get the user from the service file
SERVICE_USER=$(grep "^User=" "$SERVICE_FILE" | cut -d'=' -f2)
if [ -z "$SERVICE_USER" ]; then
    echo -e "${RED}✗${NC} Could not determine user from service file"
    exit 1
fi

USER_HOME=$(eval echo ~$SERVICE_USER)
INSTALL_DIR="${USER_HOME}/theiacast-client"

echo ""
echo -e "${BLUE}Updating service file...${NC}"

# Stop the service
echo "Stopping service..."
systemctl stop theiacast-client.service || true

# Backup old service file
cp "$SERVICE_FILE" "${SERVICE_FILE}.backup.$(date +%Y%m%d-%H%M%S)"
echo -e "${GREEN}✓${NC} Backed up old service file"

# Write new service file (without graphical.target)
cat > ${SERVICE_FILE} << EOF
[Unit]
Description=TheiaCast Kiosk Client
After=network-online.target
Wants=network-online.target

[Service]
Type=simple
User=${SERVICE_USER}
Group=${SERVICE_USER}
WorkingDirectory=${INSTALL_DIR}
Environment="NODE_ENV=production"
Environment="DISPLAY=:0"
Environment="HOME=${USER_HOME}"
Environment="XAUTHORITY=${USER_HOME}/.Xauthority"
EnvironmentFile=${INSTALL_DIR}/.env
ExecStart=/usr/bin/node ${INSTALL_DIR}/index.js
Restart=always
RestartSec=10
KillMode=control-group
KillSignal=SIGTERM
TimeoutStopSec=10

[Install]
WantedBy=multi-user.target
EOF

echo -e "${GREEN}✓${NC} Updated service file"

# Reload systemd
echo "Reloading systemd daemon..."
systemctl daemon-reload
echo -e "${GREEN}✓${NC} Reloaded systemd"

# Re-enable service (in case WantedBy changed)
systemctl enable theiacast-client.service
echo -e "${GREEN}✓${NC} Re-enabled service"

# Start the service
echo "Starting service..."
systemctl start theiacast-client.service
echo -e "${GREEN}✓${NC} Started service"

echo ""
echo -e "${GREEN}✓✓✓ Fix completed successfully! ✓✓✓${NC}"
echo ""
echo "The systemd ordering cycle should now be resolved."
echo "You can verify with: sudo journalctl -u theiacast-client -f"
echo ""
echo "On next reboot, the service should start without ordering cycle warnings."
