"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.websocketClient = exports.DeviceStatusValues = void 0;
const ws_1 = __importDefault(require("ws"));
const config_1 = require("./config");
const logger_1 = require("./logger");
// Event and enum values (since we can't import enums from CommonJS shared package)
const ServerToClientEventValues = {
    CONTENT_UPDATE: 'content:update',
    DISPLAY_NAVIGATE: 'display:navigate',
    SCREENSHOT_REQUEST: 'screenshot:request',
    CONFIG_UPDATE: 'config:update',
    DEVICE_RESTART: 'device:restart',
    DISPLAY_REFRESH: 'display:refresh',
    REMOTE_CLICK: 'remote:click',
    REMOTE_TYPE: 'remote:type',
    REMOTE_KEY: 'remote:key',
    REMOTE_SCROLL: 'remote:scroll',
    STREAM_START: 'stream:start',
    STREAM_STOP: 'stream:stop',
    PLAYLIST_PAUSE: 'playlist:pause',
    PLAYLIST_RESUME: 'playlist:resume',
    PLAYLIST_NEXT: 'playlist:next',
    PLAYLIST_PREVIOUS: 'playlist:previous',
    PLAYLIST_BROADCAST_START: 'playlist:broadcast:start',
    PLAYLIST_BROADCAST_END: 'playlist:broadcast:end',
};
const ClientToServerEventValues = {
    DEVICE_REGISTER: 'device:register',
    SCREENSHOT_UPLOAD: 'screenshot:upload',
    HEALTH_REPORT: 'health:report',
    DEVICE_STATUS: 'device:status',
    ERROR_REPORT: 'error:report',
    PLAYBACK_STATE_UPDATE: 'playback:state:update',
};
exports.DeviceStatusValues = {
    ONLINE: 'online',
    OFFLINE: 'offline',
    ERROR: 'error',
};
class WebSocketClient {
    constructor() {
        this.socket = null;
        this.reconnectAttempts = 0;
        this.maxReconnectAttempts = 1000000; // Effectively infinite
        this.isConnected = false;
    }
    connect() {
        if (this.socket?.readyState === ws_1.default.OPEN) {
            logger_1.logger.warn('WebSocket already connected');
            return;
        }
        const wsUrl = `${config_1.config.serverUrl.replace(/^http/, 'ws')}/ws?role=device&token=${encodeURIComponent(config_1.config.deviceToken)}`;
        logger_1.logger.info(`Connecting to server: ${wsUrl}`);
        this.socket = new ws_1.default(wsUrl);
        this.setupEventHandlers();
    }
    setupEventHandlers() {
        if (!this.socket)
            return;
        this.socket.on('open', () => {
            logger_1.logger.info('✅ Connected to server');
            this.isConnected = true;
            this.reconnectAttempts = 0;
            this.registerDevice();
        });
        this.socket.on('close', (code, reason) => {
            logger_1.logger.warn(`❌ Disconnected from server: code=${code} reason=${reason.toString()}`);
            this.isConnected = false;
            if (this.reconnectAttempts < this.maxReconnectAttempts) {
                this.reconnectAttempts++;
                // Exponential backoff: 5s, 10s, 20s, capped at 60s
                const delay = Math.min(5000 * Math.pow(2, this.reconnectAttempts - 1), 60000);
                logger_1.logger.info(`Reconnecting in ${delay / 1000}s (attempt ${this.reconnectAttempts})...`);
                setTimeout(() => this.connect(), delay);
            }
            else {
                logger_1.logger.error('Max reconnection attempts reached. Stopping...');
            }
        });
        this.socket.on('error', (error) => {
            logger_1.logger.error(`Connection error: ${String(error?.message || error)}`);
        });
        this.socket.on('message', (data) => {
            try {
                const msg = JSON.parse(data.toString());
                const evt = msg.event;
                const payload = msg.payload;
                switch (evt) {
                    case ServerToClientEventValues.CONTENT_UPDATE:
                        this.contentUpdateCallback?.(payload);
                        break;
                    case ServerToClientEventValues.DISPLAY_NAVIGATE:
                        this.displayNavigateCallback?.(payload);
                        break;
                    case ServerToClientEventValues.SCREENSHOT_REQUEST:
                        this.screenshotRequestCallback?.(payload);
                        break;
                    case ServerToClientEventValues.CONFIG_UPDATE:
                        this.configUpdateCallback?.(payload);
                        break;
                    case ServerToClientEventValues.DEVICE_RESTART:
                        this.deviceRestartCallback?.(payload);
                        break;
                    case ServerToClientEventValues.DISPLAY_REFRESH:
                        this.displayRefreshCallback?.(payload);
                        break;
                    case ServerToClientEventValues.REMOTE_CLICK:
                        this.remoteClickCallback?.(payload);
                        break;
                    case ServerToClientEventValues.REMOTE_TYPE:
                        this.remoteTypeCallback?.(payload);
                        break;
                    case ServerToClientEventValues.REMOTE_KEY:
                        this.remoteKeyCallback?.(payload);
                        break;
                    case ServerToClientEventValues.REMOTE_SCROLL:
                        this.remoteScrollCallback?.(payload);
                        break;
                    case ServerToClientEventValues.STREAM_START:
                        this.streamStartCallback?.(payload);
                        break;
                    case ServerToClientEventValues.STREAM_STOP:
                        this.streamStopCallback?.(payload);
                        break;
                    case ServerToClientEventValues.PLAYLIST_PAUSE:
                        this.playlistPauseCallback?.(payload);
                        break;
                    case ServerToClientEventValues.PLAYLIST_RESUME:
                        this.playlistResumeCallback?.(payload);
                        break;
                    case ServerToClientEventValues.PLAYLIST_NEXT:
                        this.playlistNextCallback?.(payload);
                        break;
                    case ServerToClientEventValues.PLAYLIST_PREVIOUS:
                        this.playlistPreviousCallback?.(payload);
                        break;
                    case ServerToClientEventValues.PLAYLIST_BROADCAST_START:
                        this.playlistBroadcastStartCallback?.(payload);
                        break;
                    case ServerToClientEventValues.PLAYLIST_BROADCAST_END:
                        this.playlistBroadcastEndCallback?.(payload);
                        break;
                    case 'screencast:start':
                        this.screencastStartCallback?.();
                        break;
                    case 'screencast:stop':
                        this.screencastStopCallback?.();
                        break;
                }
            }
            catch (err) {
                logger_1.logger.error('Failed to parse WS message', err?.message || String(err));
            }
        });
    }
    registerDevice() {
        const payload = { token: config_1.config.deviceToken };
        this.send(ClientToServerEventValues.DEVICE_REGISTER, payload);
        logger_1.logger.debug('Device registration sent');
    }
    disconnect() {
        if (this.socket) {
            try {
                this.socket.close();
            }
            catch { }
            this.socket = null;
            this.isConnected = false;
            logger_1.logger.info('WebSocket disconnected');
        }
    }
    getConnectionStatus() {
        return this.isConnected;
    }
    // Event emitters to server
    sendHealthReport(health) {
        if (!this.isConnected) {
            logger_1.logger.warn('Cannot send health report: not connected');
            return;
        }
        const payload = {
            cpu: health.cpuUsage ?? 0,
            mem: health.memoryUsage ?? 0,
            disk: health.diskUsage ?? 0,
            ts: new Date().toISOString(),
        };
        this.send(ClientToServerEventValues.HEALTH_REPORT, payload);
        logger_1.logger.debug('Health report sent', payload);
    }
    sendDeviceStatus(status, message) {
        if (!this.isConnected && status !== exports.DeviceStatusValues.OFFLINE) {
            logger_1.logger.warn('Cannot send device status: not connected');
            return;
        }
        const payload = { status, message };
        this.send(ClientToServerEventValues.DEVICE_STATUS, payload);
        logger_1.logger.debug('Device status sent', payload);
    }
    sendErrorReport(error, stack, context) {
        if (!this.isConnected) {
            logger_1.logger.warn('Cannot send error report: not connected');
            return;
        }
        const payload = { error, stack, context };
        this.send(ClientToServerEventValues.ERROR_REPORT, payload);
        logger_1.logger.error('Error report sent', payload);
    }
    sendScreenshot(image, currentUrl) {
        if (!this.isConnected) {
            logger_1.logger.warn('Cannot send screenshot: not connected');
            return;
        }
        const payload = { image, currentUrl };
        this.send(ClientToServerEventValues.SCREENSHOT_UPLOAD, payload);
        logger_1.logger.debug('Screenshot sent');
    }
    sendScreencastFrame(frameData) {
        if (!this.isConnected) {
            return; // Silently skip if not connected (too verbose to log every frame)
        }
        this.send('screencast:frame', frameData);
    }
    sendPlaybackState(state) {
        if (!this.isConnected) {
            logger_1.logger.warn('Cannot send playback state: not connected');
            return;
        }
        this.send(ClientToServerEventValues.PLAYBACK_STATE_UPDATE, state);
        logger_1.logger.debug('Playback state sent', state);
    }
    send(event, payload) {
        if (!this.isConnected || !this.socket || this.socket.readyState !== ws_1.default.OPEN) {
            logger_1.logger.warn(`Cannot send ${event}: not connected`);
            return;
        }
        const msg = JSON.stringify({ event, payload });
        this.socket.send(msg);
    }
    // Event callback setters
    onContentUpdate(callback) {
        this.contentUpdateCallback = callback;
    }
    onDisplayNavigate(callback) {
        this.displayNavigateCallback = callback;
    }
    onScreenshotRequest(callback) {
        this.screenshotRequestCallback = callback;
    }
    onConfigUpdate(callback) {
        this.configUpdateCallback = callback;
    }
    onDeviceRestart(callback) {
        this.deviceRestartCallback = callback;
    }
    onDisplayRefresh(callback) {
        this.displayRefreshCallback = callback;
    }
    onRemoteClick(callback) {
        this.remoteClickCallback = callback;
    }
    onRemoteType(callback) {
        this.remoteTypeCallback = callback;
    }
    onRemoteKey(callback) {
        this.remoteKeyCallback = callback;
    }
    onRemoteScroll(callback) {
        this.remoteScrollCallback = callback;
    }
    onStreamStart(callback) {
        this.streamStartCallback = callback;
    }
    onStreamStop(callback) {
        this.streamStopCallback = callback;
    }
    onScreencastStart(callback) {
        this.screencastStartCallback = callback;
    }
    onScreencastStop(callback) {
        this.screencastStopCallback = callback;
    }
    onPlaylistPause(callback) {
        this.playlistPauseCallback = callback;
    }
    onPlaylistResume(callback) {
        this.playlistResumeCallback = callback;
    }
    onPlaylistNext(callback) {
        this.playlistNextCallback = callback;
    }
    onPlaylistPrevious(callback) {
        this.playlistPreviousCallback = callback;
    }
    onPlaylistBroadcastStart(callback) {
        this.playlistBroadcastStartCallback = callback;
    }
    onPlaylistBroadcastEnd(callback) {
        this.playlistBroadcastEndCallback = callback;
    }
}
exports.websocketClient = new WebSocketClient();
//# sourceMappingURL=websocket.js.map