"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.screenshotManager = void 0;
const config_1 = require("./config");
const logger_1 = require("./logger");
const websocket_1 = require("./websocket");
class ScreenshotManager {
    constructor() {
        this.intervalId = null;
        this.isRunning = false;
        this.page = null;
        this.lastUrl = null;
        this.lastSentAt = 0;
    }
    setPage(page) {
        this.page = page;
        logger_1.logger.debug('Screenshot manager page reference set');
    }
    isCurrentlyRunning() {
        return this.isRunning;
    }
    start() {
        if (this.isRunning) {
            return;
        }
        if (!this.page) {
            logger_1.logger.warn('Cannot start screenshot manager: no page reference set');
            return;
        }
        const interval = config_1.config.screenshotInterval || 30000; // Default 30 seconds
        logger_1.logger.info(`Starting screenshot manager with ${interval}ms interval`);
        this.isRunning = true;
        // Send periodic screenshots for monitoring
        // Especially important for single-item playlists that never change
        this.intervalId = setInterval(() => {
            this.captureAndSendScreenshot();
        }, interval);
        // Send initial screenshot
        this.captureAndSendScreenshot();
    }
    stop() {
        if (this.intervalId) {
            clearInterval(this.intervalId);
            this.intervalId = null;
            this.isRunning = false;
            logger_1.logger.info('Screenshot manager stopped');
        }
    }
    async captureAndSendScreenshot() {
        if (!this.page) {
            logger_1.logger.warn('Cannot capture screenshot: no page reference');
            return;
        }
        try {
            // Check if page is in a valid state before attempting screenshot
            const currentUrl = this.page.url();
            // Skip if page is closed, about:blank, or in invalid state
            if (!currentUrl || currentUrl === 'about:blank') {
                logger_1.logger.debug('Skipping screenshot: page not loaded or in about:blank state');
                return;
            }
            // Check if page is still open by trying to access it
            let isPageOpen = false;
            try {
                isPageOpen = !this.page.isClosed();
            }
            catch (e) {
                logger_1.logger.warn('Page state check failed, assuming closed');
                return;
            }
            if (!isPageOpen) {
                logger_1.logger.warn('Skipping screenshot: page is closed');
                return;
            }
            // Change-triggered capture: if URL changed since last capture, send immediately
            const urlChanged = this.lastUrl !== currentUrl;
            // Throttle rapid repeats: avoid >1/sec even on fast changes
            const now = Date.now();
            if (!urlChanged && now - this.lastSentAt < 900) {
                logger_1.logger.debug('Skipping screenshot: recently sent');
                return;
            }
            logger_1.logger.debug('Capturing screenshot...');
            const screenshot = await this.page.screenshot({
                encoding: 'base64',
                type: 'jpeg',
                quality: 50, // Reduced quality to save bandwidth
                fullPage: false,
            });
            websocket_1.websocketClient.sendScreenshot(screenshot, currentUrl);
            this.lastUrl = currentUrl;
            this.lastSentAt = now;
            logger_1.logger.info(`Screenshot captured and sent (${urlChanged ? 'url change' : 'periodic'})`);
        }
        catch (error) {
            // Check if it's a "session closed" error - this is expected during navigation
            const isSessionClosed = error.message && (error.message.includes('Session closed') ||
                error.message.includes('Target closed') ||
                error.message.includes('Page has been closed'));
            if (isSessionClosed) {
                logger_1.logger.debug('Screenshot skipped: page session closed (likely during navigation)');
                return; // Don't report this as an error - it's expected during transitions
            }
            // For other errors, log and report
            logger_1.logger.error('Failed to capture screenshot:', error.message);
            try {
                websocket_1.websocketClient.sendErrorReport('Screenshot capture failed', error.stack, { url: this.page?.url() || 'unknown' });
            }
            catch (reportError) {
                // Ignore errors when trying to report errors
                logger_1.logger.warn('Could not send error report:', reportError);
            }
        }
    }
    async captureOnDemand() {
        logger_1.logger.info('On-demand screenshot requested');
        await this.captureAndSendScreenshot();
    }
}
exports.screenshotManager = new ScreenshotManager();
//# sourceMappingURL=screenshot.js.map