#!/usr/bin/env node
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const config_1 = require("./config");
const logger_1 = require("./logger");
const websocket_1 = require("./websocket");
const display_1 = require("./display");
const health_1 = require("./health");
const screenshot_1 = require("./screenshot");
const playlist_executor_1 = require("./playlist-executor");
class KioskClient {
    constructor() {
        this.isShuttingDown = false;
    }
    async start() {
        try {
            logger_1.logger.info('===========================================');
            logger_1.logger.info('  Kiosk Digital Signage Client Starting');
            logger_1.logger.info('===========================================');
            logger_1.logger.info(`Server URL: ${config_1.config.serverUrl}`);
            logger_1.logger.info(`Display: ${config_1.config.displayWidth}x${config_1.config.displayHeight}`);
            logger_1.logger.info(`Kiosk Mode: ${config_1.config.kioskMode ? 'Enabled' : 'Disabled'}`);
            logger_1.logger.info('===========================================');
            // Setup signal handlers for graceful shutdown
            this.setupSignalHandlers();
            // Setup WebSocket handlers before connecting
            this.setupWebSocketHandlers();
            // Initialize display controller with current config
            // (Config updates from server will trigger display restart if needed)
            await display_1.displayController.initialize();
            // Connect to backend via WebSocket
            // This will trigger device registration and receive playlist + config
            websocket_1.websocketClient.connect();
            // Start health monitoring
            health_1.healthMonitor.start();
            // NOTE: Screenshot manager is started by playlist executor based on playlist type
            // - Single-item playlists: periodic screenshots every 30s
            // - Multi-item playlists: screenshots on each rotation only
            logger_1.logger.info('✅ Kiosk client started successfully');
            logger_1.logger.info('Waiting for playlist from server...');
        }
        catch (error) {
            logger_1.logger.error('Failed to start kiosk client:', error.message);
            logger_1.logger.error(error.stack);
            process.exit(1);
        }
    }
    setupWebSocketHandlers() {
        // Content update handler
        websocket_1.websocketClient.onContentUpdate(async (payload) => {
            logger_1.logger.info(`Received content update with ${payload.items.length} items`);
            await playlist_executor_1.playlistExecutor.loadPlaylist(payload.items);
            playlist_executor_1.playlistExecutor.start();
        });
        // Playlist control handlers
        websocket_1.websocketClient.onPlaylistPause(() => {
            logger_1.logger.info('Playlist pause requested');
            playlist_executor_1.playlistExecutor.pause();
        });
        websocket_1.websocketClient.onPlaylistResume(() => {
            logger_1.logger.info('Playlist resume requested');
            playlist_executor_1.playlistExecutor.resume();
        });
        websocket_1.websocketClient.onPlaylistNext((payload) => {
            logger_1.logger.info('Playlist next requested');
            playlist_executor_1.playlistExecutor.next(payload.respectConstraints !== false);
        });
        websocket_1.websocketClient.onPlaylistPrevious((payload) => {
            logger_1.logger.info('Playlist previous requested');
            playlist_executor_1.playlistExecutor.previous(payload.respectConstraints !== false);
        });
        websocket_1.websocketClient.onPlaylistBroadcastStart((payload) => {
            logger_1.logger.info(`>>> Received playlist:broadcast:start event`);
            if (payload.type === 'url' && payload.url) {
                logger_1.logger.info(`Playlist broadcast start (URL): ${payload.url} for ${payload.duration || 0}ms`);
                playlist_executor_1.playlistExecutor.startBroadcast(payload.type, payload.url, undefined, payload.duration || 0);
            }
            else if (payload.type === 'message' && payload.message) {
                logger_1.logger.info(`Playlist broadcast start (Message): ${payload.message} for ${payload.duration || 0}ms, Background: ${payload.background?.length || 0} chars, Logo: ${payload.logo?.length || 0} chars, Position: ${payload.logoPosition}`);
                playlist_executor_1.playlistExecutor.startBroadcast(payload.type, undefined, payload.message, payload.duration || 0, payload.background, payload.logo, payload.logoPosition);
            }
            else if ((payload.type === 'image' || payload.type === 'video') && payload.mediaData) {
                logger_1.logger.info(`Playlist broadcast start (${payload.type}): Media size ${payload.mediaData.length} chars for ${payload.duration || 0}ms`);
                playlist_executor_1.playlistExecutor.startBroadcast(payload.type, undefined, undefined, payload.duration || 0, undefined, undefined, undefined, payload.mediaData);
            }
            else {
                logger_1.logger.error('Invalid broadcast payload:', payload);
            }
        });
        websocket_1.websocketClient.onPlaylistBroadcastEnd(() => {
            logger_1.logger.info('Playlist broadcast end requested');
            playlist_executor_1.playlistExecutor.endBroadcast();
        });
        // Display navigate handler
        websocket_1.websocketClient.onDisplayNavigate((payload) => {
            logger_1.logger.info(`Navigating to: ${payload.url}`);
            display_1.displayController.navigateTo(payload.url, payload.duration);
        });
        // Screenshot request handler
        websocket_1.websocketClient.onScreenshotRequest((payload) => {
            logger_1.logger.info('Screenshot requested by server');
            screenshot_1.screenshotManager.captureOnDemand();
        });
        // Server-controlled stream start/stop
        websocket_1.websocketClient.onStreamStart(async () => {
            logger_1.logger.info('Stream start requested by server');
            await display_1.displayController.startScreencast();
        });
        websocket_1.websocketClient.onStreamStop(async () => {
            logger_1.logger.info('Stream stop requested by server');
            await display_1.displayController.stopScreencast();
        });
        // Config update handler
        websocket_1.websocketClient.onConfigUpdate(async (payload) => {
            logger_1.logger.info('Configuration update received', payload);
            // Track if display config changed (requires restart)
            let displayConfigChanged = false;
            // Update config with new values
            if (payload.screenshotInterval) {
                config_1.configManager.update({ screenshotInterval: payload.screenshotInterval });
                // Restart screenshot manager only if it's currently running
                // (Single-item playlists use periodic screenshots, multi-item don't)
                const wasRunning = screenshot_1.screenshotManager.isCurrentlyRunning();
                if (wasRunning) {
                    screenshot_1.screenshotManager.stop();
                    screenshot_1.screenshotManager.start();
                }
            }
            if (payload.healthCheckInterval) {
                config_1.configManager.update({ healthCheckInterval: payload.healthCheckInterval });
                health_1.healthMonitor.stop();
                health_1.healthMonitor.start();
            }
            // Handle display configuration updates
            const currentConfig = config_1.configManager.get();
            if (payload.displayWidth !== undefined && payload.displayWidth !== currentConfig.displayWidth) {
                config_1.configManager.update({ displayWidth: payload.displayWidth });
                displayConfigChanged = true;
            }
            if (payload.displayHeight !== undefined && payload.displayHeight !== currentConfig.displayHeight) {
                config_1.configManager.update({ displayHeight: payload.displayHeight });
                displayConfigChanged = true;
            }
            if (payload.kioskMode !== undefined && payload.kioskMode !== currentConfig.kioskMode) {
                config_1.configManager.update({ kioskMode: payload.kioskMode });
                displayConfigChanged = true;
            }
            // Restart display if display config changed
            if (displayConfigChanged) {
                logger_1.logger.info('Display configuration changed, restarting display...');
                await display_1.displayController.restart();
                logger_1.logger.info('Display restarted with new configuration');
            }
        });
        // Device restart handler
        websocket_1.websocketClient.onDeviceRestart((payload) => {
            logger_1.logger.warn('Device restart requested by server', payload);
            this.restart();
        });
        // Display refresh handler
        websocket_1.websocketClient.onDisplayRefresh((payload) => {
            logger_1.logger.info('Display refresh requested', payload);
            display_1.displayController.refresh(payload.force);
        });
        // Remote control handlers
        websocket_1.websocketClient.onRemoteClick((payload) => {
            logger_1.logger.info(`Remote click at (${payload.x}, ${payload.y})`);
            display_1.displayController.remoteClick(payload.x, payload.y, payload.button);
        });
        websocket_1.websocketClient.onRemoteType((payload) => {
            logger_1.logger.info(`Remote type: ${payload.text.substring(0, 20)}...`);
            display_1.displayController.remoteType(payload.text, payload.selector);
        });
        websocket_1.websocketClient.onRemoteKey((payload) => {
            logger_1.logger.info(`Remote key: ${payload.key}`);
            display_1.displayController.remoteKey(payload.key, payload.modifiers);
        });
        websocket_1.websocketClient.onRemoteScroll((payload) => {
            logger_1.logger.info('Remote scroll requested');
            display_1.displayController.remoteScroll(payload.x, payload.y, payload.deltaX, payload.deltaY);
        });
        // Screencast control handlers
        websocket_1.websocketClient.onScreencastStart(() => {
            logger_1.logger.info('Admin requested screencast start');
            display_1.displayController.startScreencast();
        });
        websocket_1.websocketClient.onScreencastStop(() => {
            logger_1.logger.info('Admin requested screencast stop');
            display_1.displayController.stopScreencast();
        });
    }
    setupSignalHandlers() {
        const signals = ['SIGINT', 'SIGTERM', 'SIGQUIT'];
        signals.forEach((signal) => {
            process.on(signal, () => {
                logger_1.logger.info(`\nReceived ${signal}, shutting down gracefully...`);
                this.shutdown();
            });
        });
        process.on('uncaughtException', (error) => {
            logger_1.logger.error('Uncaught exception:', error.message);
            logger_1.logger.error(error.stack || '');
            websocket_1.websocketClient.sendErrorReport('Uncaught exception', error.stack, { message: error.message });
        });
        process.on('unhandledRejection', (reason) => {
            logger_1.logger.error('Unhandled rejection:', reason);
            websocket_1.websocketClient.sendErrorReport('Unhandled rejection', reason?.stack, { reason: String(reason) });
        });
    }
    async restart() {
        try {
            logger_1.logger.warn('Restarting kiosk client...');
            // Stop all services
            playlist_executor_1.playlistExecutor.stop();
            screenshot_1.screenshotManager.stop();
            health_1.healthMonitor.stop();
            // Restart display
            await display_1.displayController.restart();
            // Restart services
            health_1.healthMonitor.start();
            // Restart playlist executor if it has a playlist
            // (Playlist executor will start screenshot manager based on playlist type)
            if (playlist_executor_1.playlistExecutor.hasPlaylist()) {
                playlist_executor_1.playlistExecutor.start();
            }
            logger_1.logger.info('✅ Kiosk client restarted successfully');
        }
        catch (error) {
            logger_1.logger.error('Failed to restart:', error.message);
            websocket_1.websocketClient.sendErrorReport('Restart failed', error.stack);
        }
    }
    async shutdown() {
        if (this.isShuttingDown) {
            return;
        }
        this.isShuttingDown = true;
        logger_1.logger.info('Shutting down kiosk client...');
        // Send offline status
        websocket_1.websocketClient.sendDeviceStatus(websocket_1.DeviceStatusValues.OFFLINE, 'Client shutting down');
        // Stop all services
        playlist_executor_1.playlistExecutor.stop();
        screenshot_1.screenshotManager.stop();
        health_1.healthMonitor.stop();
        // Disconnect WebSocket
        websocket_1.websocketClient.disconnect();
        // Shutdown display
        await display_1.displayController.shutdown();
        logger_1.logger.info('✅ Kiosk client shut down successfully');
        process.exit(0);
    }
}
// Start the client
const client = new KioskClient();
client.start().catch((error) => {
    console.error('Fatal error:', error);
    process.exit(1);
});
//# sourceMappingURL=index.js.map