"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || (function () {
    var ownKeys = function(o) {
        ownKeys = Object.getOwnPropertyNames || function (o) {
            var ar = [];
            for (var k in o) if (Object.prototype.hasOwnProperty.call(o, k)) ar[ar.length] = k;
            return ar;
        };
        return ownKeys(o);
    };
    return function (mod) {
        if (mod && mod.__esModule) return mod;
        var result = {};
        if (mod != null) for (var k = ownKeys(mod), i = 0; i < k.length; i++) if (k[i] !== "default") __createBinding(result, mod, k[i]);
        __setModuleDefault(result, mod);
        return result;
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.contentCacheManager = exports.ContentCacheManager = exports.CacheStatus = void 0;
const fs = __importStar(require("fs"));
const path = __importStar(require("path"));
const stream = __importStar(require("stream"));
const http = __importStar(require("http"));
const https = __importStar(require("https"));
const util_1 = require("util");
const config_1 = require("./config");
const logger_1 = require("./logger");
const pipeline = (0, util_1.promisify)(stream.pipeline);
var CacheStatus;
(function (CacheStatus) {
    CacheStatus["NotCached"] = "not_cached";
    CacheStatus["Downloading"] = "downloading";
    CacheStatus["Ready"] = "ready";
    CacheStatus["Error"] = "error";
})(CacheStatus || (exports.CacheStatus = CacheStatus = {}));
class ContentCacheManager {
    constructor() {
        this.activeDownloads = new Set();
        this.cacheEntries = new Map();
        // Use a cache directory in the user's home or app data
        const homeDir = process.env.HOME || process.env.USERPROFILE || '.';
        this.cacheDir = path.join(homeDir, '.pds-cache');
        this.ensureCacheDir();
    }
    ensureCacheDir() {
        if (!fs.existsSync(this.cacheDir)) {
            fs.mkdirSync(this.cacheDir, { recursive: true });
        }
    }
    getLocalPath(url) {
        const relativePath = this.getRelativePathFromUrl(url);
        if (!relativePath)
            return null;
        const localPath = path.join(this.cacheDir, relativePath);
        if (fs.existsSync(localPath)) {
            return localPath;
        }
        return null;
    }
    getCacheStatus(url) {
        const relativePath = this.getRelativePathFromUrl(url);
        if (!relativePath)
            return CacheStatus.NotCached;
        const entry = this.cacheEntries.get(url);
        if (entry) {
            return entry.status;
        }
        // Check if file exists
        const localPath = path.join(this.cacheDir, relativePath);
        if (fs.existsSync(localPath)) {
            return CacheStatus.Ready;
        }
        return CacheStatus.NotCached;
    }
    async waitForCache(url, maxWaitMs = 300000) {
        if (!this.isCacheable(url)) {
            return null; // Not a cacheable URL
        }
        const startTime = Date.now();
        const checkInterval = 1000; // Check every second
        while (Date.now() - startTime < maxWaitMs) {
            const status = this.getCacheStatus(url);
            if (status === CacheStatus.Ready) {
                return this.getLocalPath(url);
            }
            else if (status === CacheStatus.Error) {
                logger_1.logger.warn(`Video caching failed for: ${url}`);
                return null;
            }
            // Still downloading, wait a bit
            await new Promise(resolve => setTimeout(resolve, checkInterval));
        }
        logger_1.logger.warn(`Timeout waiting for video cache: ${url}`);
        return null;
    }
    async syncPlaylist(items) {
        // Run in background to not block playback
        this.syncPlaylistInternal(items).catch(err => {
            logger_1.logger.error(`Background sync failed: ${err.message}`);
        });
    }
    async syncPlaylistInternal(items) {
        logger_1.logger.info('Starting background sync of playlist content...');
        const config = config_1.configManager.get();
        const baseUrl = config.serverUrl.endsWith('/') ? config.serverUrl.slice(0, -1) : config.serverUrl;
        const activeFiles = new Set();
        for (const item of items) {
            if (!item.content || !item.content.url)
                continue;
            let url = item.content.url;
            if (!this.isCacheable(url))
                continue;
            const relativePath = this.getRelativePathFromUrl(url);
            if (!relativePath)
                continue;
            activeFiles.add(relativePath);
            const localPath = path.join(this.cacheDir, relativePath);
            const localDir = path.dirname(localPath);
            // Ensure directory exists
            if (!fs.existsSync(localDir)) {
                fs.mkdirSync(localDir, { recursive: true });
            }
            // Check if file exists or is already downloading
            // Use a unique key for active downloads to avoid collisions
            const downloadKey = relativePath;
            if (!fs.existsSync(localPath) && !this.activeDownloads.has(downloadKey)) {
                const fullUrl = url.startsWith('http') ? url : `${baseUrl}${url.startsWith('/') ? '' : '/'}${url}`;
                // Mark as downloading
                this.cacheEntries.set(url, {
                    relativePath,
                    status: CacheStatus.Downloading
                });
                this.activeDownloads.add(downloadKey);
                try {
                    if (relativePath.endsWith('index.html')) {
                        await this.downloadVideoWrapper(fullUrl, localPath, baseUrl);
                    }
                    else {
                        logger_1.logger.info(`Downloading ${path.basename(localPath)}...`);
                        await this.downloadFile(fullUrl, localPath);
                        logger_1.logger.info(`✅ Downloaded ${path.basename(localPath)}`);
                    }
                    // Mark as ready
                    this.cacheEntries.set(url, {
                        relativePath,
                        status: CacheStatus.Ready
                    });
                }
                catch (err) {
                    logger_1.logger.error(`Failed to download ${fullUrl}: ${err.message}`);
                    // Mark as error
                    this.cacheEntries.set(url, {
                        relativePath,
                        status: CacheStatus.Error,
                        error: err.message
                    });
                }
                finally {
                    this.activeDownloads.delete(downloadKey);
                }
            }
            else if (fs.existsSync(localPath)) {
                // Already cached
                this.cacheEntries.set(url, {
                    relativePath,
                    status: CacheStatus.Ready
                });
            }
        }
        this.cleanup(activeFiles);
    }
    cleanup(activeFiles) {
        try {
            // Helper to recursively clean
            const cleanDir = (dir, relativeRoot) => {
                const entries = fs.readdirSync(dir, { withFileTypes: true });
                let isEmpty = true;
                for (const entry of entries) {
                    const fullPath = path.join(dir, entry.name);
                    const relPath = path.join(relativeRoot, entry.name);
                    if (entry.isDirectory()) {
                        if (cleanDir(fullPath, relPath)) {
                            fs.rmdirSync(fullPath);
                        }
                        else {
                            isEmpty = false;
                        }
                    }
                    else {
                        // Keep if in activeFiles OR if it's a video file in an active wrapper folder
                        // If activeFiles has "GUID/index.html", we keep "GUID/video.mp4" too.
                        const parentDir = path.dirname(relPath);
                        const parentIndex = path.join(parentDir, 'index.html');
                        const keep = activeFiles.has(relPath) ||
                            (activeFiles.has(parentIndex) && !entry.name.endsWith('.tmp'));
                        if (!keep && !entry.name.endsWith('.tmp')) {
                            logger_1.logger.info(`Removing unused cache file: ${relPath}`);
                            try {
                                fs.unlinkSync(fullPath);
                            }
                            catch (e) { /* ignore */ }
                        }
                        else {
                            isEmpty = false;
                        }
                    }
                }
                return isEmpty;
            };
            cleanDir(this.cacheDir, '');
        }
        catch (e) {
            logger_1.logger.error(`Cache cleanup failed: ${e.message}`);
        }
    }
    isCacheable(url) {
        const lower = url.toLowerCase();
        const ext = path.extname(lower);
        if (['.mp4', '.webm', '.mkv', '.avi', '.mov'].includes(ext))
            return true;
        if (lower.includes('/videos/') && lower.endsWith('/index.html'))
            return true;
        return false;
    }
    getRelativePathFromUrl(url) {
        try {
            // Check for new video wrapper format: .../videos/{guid}/index.html
            const videoWrapperMatch = url.match(/\/videos\/([a-f0-9-]+)\/index\.html$/i);
            if (videoWrapperMatch) {
                return path.join(videoWrapperMatch[1], 'index.html');
            }
            // Handle standard /path/to/file.mp4
            const parts = url.split('/');
            return parts[parts.length - 1];
        }
        catch {
            return null;
        }
    }
    async downloadVideoWrapper(url, localHtmlPath, baseUrl) {
        const localDir = path.dirname(localHtmlPath);
        // 1. Download HTML content
        const htmlContent = await this.fetchText(url);
        // 2. Find video src
        const srcMatch = htmlContent.match(/<video[^>]+src="([^"]+)"/);
        if (!srcMatch) {
            throw new Error('No video src found in wrapper HTML');
        }
        const videoFilename = srcMatch[1];
        const videoUrl = new URL(videoFilename, url).toString();
        const localVideoPath = path.join(localDir, videoFilename);
        // 3. Download video if missing
        if (!fs.existsSync(localVideoPath)) {
            logger_1.logger.info(`Downloading wrapper video: ${videoFilename}...`);
            await this.downloadFile(videoUrl, localVideoPath);
            logger_1.logger.info(`✅ Downloaded ${videoFilename}`);
        }
        // 4. Save HTML
        fs.writeFileSync(localHtmlPath, htmlContent);
    }
    fetchText(url) {
        return new Promise((resolve, reject) => {
            const protocol = url.startsWith('https') ? https : http;
            protocol.get(url, (res) => {
                if (res.statusCode !== 200) {
                    reject(new Error(`HTTP ${res.statusCode}`));
                    return;
                }
                let data = '';
                res.on('data', chunk => data += chunk);
                res.on('end', () => resolve(data));
            }).on('error', reject);
        });
    }
    async downloadFile(url, dest) {
        const tmpDest = `${dest}.tmp`;
        const file = fs.createWriteStream(tmpDest);
        return new Promise((resolve, reject) => {
            const protocol = url.startsWith('https') ? https : http;
            const request = protocol.get(url, (response) => {
                if (response.statusCode !== 200) {
                    fs.unlink(tmpDest, () => { }); // Delete tmp file
                    reject(new Error(`HTTP ${response.statusCode} ${response.statusMessage}`));
                    return;
                }
                const totalSize = parseInt(response.headers['content-length'] || '0', 10);
                let downloadedSize = 0;
                let lastLoggedPercent = 0;
                response.on('data', (chunk) => {
                    downloadedSize += chunk.length;
                    if (totalSize > 0) {
                        const percent = Math.floor((downloadedSize / totalSize) * 100);
                        // Log every 10%
                        if (percent >= lastLoggedPercent + 10) {
                            logger_1.logger.info(`Downloading ${path.basename(dest)}: ${percent}% (${(downloadedSize / 1024 / 1024).toFixed(1)} MB)`);
                            lastLoggedPercent = percent;
                        }
                    }
                });
                response.pipe(file);
                file.on('finish', () => {
                    file.close(() => {
                        // Rename tmp to final
                        fs.rename(tmpDest, dest, (err) => {
                            if (err)
                                reject(err);
                            else
                                resolve();
                        });
                    });
                });
            });
            request.on('error', (err) => {
                fs.unlink(tmpDest, () => { }); // Delete tmp file
                reject(err);
            });
            // Set a long timeout (1 hour)
            request.setTimeout(3600000, () => {
                request.destroy();
                fs.unlink(tmpDest, () => { }); // Delete tmp file
                reject(new Error('Download timeout'));
            });
        });
    }
}
exports.ContentCacheManager = ContentCacheManager;
exports.contentCacheManager = new ContentCacheManager();
//# sourceMappingURL=content-cache.js.map